/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package org.mozilla.fenix.settings.logins.ui

import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material3.Icon
import androidx.compose.material3.Surface
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.text.input.PasswordVisualTransformation
import androidx.compose.ui.text.input.VisualTransformation
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.tooling.preview.PreviewLightDark
import androidx.compose.ui.unit.dp
import mozilla.components.compose.base.button.IconButton
import mozilla.components.compose.base.text.Text
import mozilla.components.compose.base.text.value
import mozilla.components.compose.base.textfield.TextField
import org.mozilla.fenix.theme.FirefoxTheme
import org.mozilla.fenix.theme.Theme
import mozilla.components.ui.icons.R as iconsR

/**
 * An eye trailing icon for a [TextField] that contains a password.
 * @param isPasswordVisible true when the password is revealed.
 * @param contentDescription the content description.
 * @param onTrailingIconClick invoked when pressing the eye icon
 */
@Composable
fun EyePasswordIconButton(
    isPasswordVisible: Boolean = false,
    contentDescription: Text? = null,
    onTrailingIconClick: () -> Unit,
) {
    IconButton(
        onClick = onTrailingIconClick,
        contentDescription = contentDescription?.value,
    ) {
        Icon(
            painter = if (!isPasswordVisible) {
                painterResource(id = iconsR.drawable.mozac_ic_eye_24)
            } else {
                painterResource(id = iconsR.drawable.mozac_ic_eye_slash_24)
            },
            contentDescription = null,
        )
    }
}

@PreviewLightDark
@Composable
private fun EyePasswordIconButtonPreview() {
    var isPasswordVisible by remember { mutableStateOf(false) }

    FirefoxTheme {
        Surface {
            TextField(
                value = "password",
                onValueChange = {},
                isEnabled = true,
                placeholder = "",
                errorText = "",
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(8.dp),
                label = "",
                trailingIcon = {
                    EyePasswordIconButton(
                        isPasswordVisible = isPasswordVisible,
                        onTrailingIconClick = { isPasswordVisible = !isPasswordVisible },
                    )
                },
                visualTransformation = if (isPasswordVisible) {
                    VisualTransformation.None
                } else {
                    PasswordVisualTransformation()
                },
            )
        }
    }
}

@Preview
@Composable
private fun EyePasswordIconButtonPrivatePreview() {
    var isPasswordVisible by remember { mutableStateOf(false) }

    FirefoxTheme(theme = Theme.Private) {
        Surface {
            TextField(
                value = "password",
                onValueChange = {},
                isEnabled = true,
                placeholder = "",
                errorText = "",
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(8.dp),
                label = "",
                trailingIcon = {
                    EyePasswordIconButton(
                        isPasswordVisible = isPasswordVisible,
                        onTrailingIconClick = { isPasswordVisible = !isPasswordVisible },
                    )
                },
                visualTransformation = if (isPasswordVisible) {
                    VisualTransformation.None
                } else {
                    PasswordVisualTransformation()
                },
            )
        }
    }
}
