/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

package org.mozilla.fenix.tabstray.syncedtabs

import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.map
import mozilla.components.feature.syncedtabs.view.SyncedTabsView
import mozilla.components.lib.state.helpers.AbstractBinding
import org.mozilla.fenix.tabstray.TabsTrayState
import org.mozilla.fenix.tabstray.TabsTrayStore

/**
 * An [AbstractBinding] that invokes the [onSyncNow] callback when the [TabsTrayState.syncing] is
 * set.
 *
 * This binding is useful for connecting with [SyncedTabsView.Listener].
 */
class SyncButtonBinding(
    tabsTrayStore: TabsTrayStore,
    mainDispatcher: CoroutineDispatcher = Dispatchers.Main,
    private val onSyncNow: () -> Unit,
) : AbstractBinding<TabsTrayState>(tabsTrayStore, mainDispatcher) {
    override suspend fun onState(flow: Flow<TabsTrayState>) {
        flow.map { it.syncing }
            .distinctUntilChanged()
            .collect { syncingNow ->
                if (syncingNow) {
                    onSyncNow()
                }
            }
    }
}
